<?php
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/message_logging.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

function sendEmail($to, $subject, $message, $attachments = []) {
    global $pdo;
    try {
        // Get email settings from database
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        // Check if email is enabled
        if (($settings['enable_email'] ?? '0') !== '1') {
            logMessageAttempt('email', $to, $message, 'failed', 'Email notifications are not enabled');
            throw new Exception('Email notifications are not enabled');
        }

        // Get SMTP settings
        $smtpHost = $settings['smtp_host'] ?? '';
        $smtpPort = $settings['smtp_port'] ?? '';
        $smtpUsername = $settings['smtp_username'] ?? '';
        $smtpPassword = $settings['smtp_password'] ?? '';
        $smtpEncryption = $settings['smtp_encryption'] ?? 'tls';
        $senderName = $settings['smtp_sender_name'] ?? 'POS System';

        // Validate required settings
        if (empty($smtpHost) || empty($smtpPort) || empty($smtpUsername) || empty($smtpPassword)) {
            logMessageAttempt('email', $to, $message, 'failed', 'Email settings are not properly configured');
            throw new Exception('Email settings are not properly configured');
        }

        // Create PHPMailer instance
        $mail = new PHPMailer(true);

        try {
            // Server settings
            $mail->isSMTP();
            $mail->Host = $smtpHost;
            $mail->SMTPAuth = true;
            $mail->Username = $smtpUsername;
            $mail->Password = $smtpPassword;
            $mail->SMTPSecure = $smtpEncryption;
            $mail->Port = $smtpPort;
            $mail->CharSet = 'UTF-8';

            // Recipients
            $mail->setFrom($smtpUsername, $senderName);
            $mail->addAddress($to);

            // Content
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $message;

            // Add attachments if any
            foreach ($attachments as $attachment) {
                if (isset($attachment['path']) && file_exists($attachment['path'])) {
                    $mail->addAttachment(
                        $attachment['path'],
                        $attachment['name'] ?? basename($attachment['path'])
                    );
                }
            }

            $mail->send();
            logMessageAttempt('email', $to, $message, 'success', '', 'Sent via SMTP');
            return true;
        } catch (Exception $e) {
            $error = "Email could not be sent. Mailer Error: {$mail->ErrorInfo}";
            logMessageAttempt('email', $to, $message, 'failed', $error);
            throw new Exception($error);
        }
    } catch (Exception $e) {
        error_log("Email Error: " . $e->getMessage());
        throw $e;
    }
}

// Function to test email configuration
function testEmailSettings() {
    global $pdo;
    try {
        // Get email settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        if (($settings['enable_email'] ?? '0') !== '1') {
            logMessageAttempt('email', 'test', 'Test Email', 'failed', 'Email notifications are not enabled');
            return ['success' => false, 'message' => 'Email notifications are not enabled'];
        }

        $smtpHost = $settings['smtp_host'] ?? '';
        $smtpPort = $settings['smtp_port'] ?? '';
        $smtpUsername = $settings['smtp_username'] ?? '';
        $smtpPassword = $settings['smtp_password'] ?? '';
        $smtpEncryption = $settings['smtp_encryption'] ?? 'tls';
        $testEmail = $settings['email_test_address'] ?? '';

        if (empty($smtpHost) || empty($smtpPort) || empty($smtpUsername) || empty($smtpPassword)) {
            logMessageAttempt('email', 'test', 'Test Email', 'failed', 'Email settings are not properly configured');
            return ['success' => false, 'message' => 'Email settings are not properly configured'];
        }

        if (empty($testEmail)) {
            logMessageAttempt('email', 'test', 'Test Email', 'failed', 'Test email address not configured');
            return ['success' => false, 'message' => 'Test email address not configured'];
        }

        // Send a test email
        $testSubject = "Test Email from POS System";
        $testMessage = "
            <html>
            <body>
                <h2>Test Email</h2>
                <p>This is a test email from your POS system. If you receive this, your email configuration is working correctly.</p>
                <p>Configuration details:</p>
                <ul>
                    <li>SMTP Host: {$smtpHost}</li>
                    <li>SMTP Port: {$smtpPort}</li>
                    <li>Encryption: {$smtpEncryption}</li>
                </ul>
            </body>
            </html>
        ";

        sendEmail($testEmail, $testSubject, $testMessage);
        logMessageAttempt('email', $testEmail, $testMessage, 'success', '', 'Test email sent successfully');
        return ['success' => true, 'message' => 'Test email sent successfully'];

    } catch (Exception $e) {
        logMessageAttempt('email', 'test', 'Test Email', 'failed', $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
} 